
//==============================================================================
// COMPREHENSIVE ANALYSIS FILE
// Generates all figures and tables in manuscript order
//==============================================================================

// Note: These globals are set in 0PathSetup.do
global mydata $data_folder/data.dta
global myfigs $output_folder/figures
global mytables $output_folder/tables

set more off

//==============================================================================
// MAIN TEXT - FIGURES AND TABLES
//==============================================================================


*------------------------------------------------------------------------------
// TABLE 2: The frequency of preventive wars (descriptive statistics)
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE 2: THE FREQUENCY OF PREVENTIVE WARS"
display "========================================================"
display ""

use $mydata, clear

* Calculate preventive war rate by treatment
* All rounds
preserve
collapse (mean) war_rate = reject1, by(treatment)
gen war_rate_pct = war_rate * 100

* Extract values using a loop
forvalues i = 1/5 {
	local treatment_name = word("1PreventiveWar 2RS-Contain 3RS-Commit 4RS-Choice 5DS-Contain", `i')
	local rate_label = word("PW_all RS_contain_all RS_commit_all RS_choice_all DS_contain_all", `i')
	sum war_rate_pct if treatment == "`treatment_name'", meanonly
	local `rate_label' = string(r(mean), "%2.1f")
}
restore

* Last 10 rounds
preserve
keep if period > 20
collapse (mean) war_rate = reject1, by(treatment)
gen war_rate_pct = war_rate * 100

* Extract values using a loop
forvalues i = 1/5 {
	local treatment_name = word("1PreventiveWar 2RS-Contain 3RS-Commit 4RS-Choice 5DS-Contain", `i')
	local rate_label = word("PW_last10 RS_contain_last10 RS_commit_last10 RS_choice_last10 DS_contain_last10", `i')
	sum war_rate_pct if treatment == "`treatment_name'", meanonly
	local `rate_label' = string(r(mean), "%2.1f")
}
restore

* Display table in console
display ""
display "                   All rounds    Last 10 rounds"
display "-------------------------------------------------"
display "PW-Baseline        `PW_all'%        `PW_last10'%"
display "RS-Contain         `RS_contain_all'%        `RS_contain_last10'%"
display "RS-Commit          `RS_commit_all'%        `RS_commit_last10'%"
display "RS-Choice          `RS_choice_all'%        `RS_choice_last10'%"
display "DS-Contain         `DS_contain_all'%        `DS_contain_last10'%"
display "-------------------------------------------------"

* Export to LaTeX
file open texfile using "$mytables/table2.tex", write replace text
file write texfile "\begin{table}[htb]" _n
file write texfile "    \caption{The frequency of preventive wars}" _n
file write texfile "    \label{table:preventive-war-rate}" _n
file write texfile "    \begin{center}" _n
file write texfile "        \begin{tabular}{l*2{c}}" _n
file write texfile "        \hline" _n
file write texfile "                   & All rounds & Last 10 rounds \\" _n
file write texfile "        \hline" _n
file write texfile "        PW-Baseline& `PW_all'\% & `PW_last10'\% \\" _n
file write texfile "        RS-Contain & `RS_contain_all'\% & `RS_contain_last10'\% \\" _n
file write texfile "        RS-Commit  & `RS_commit_all'\% & `RS_commit_last10'\% \\" _n
file write texfile "        RS-Choice  & `RS_choice_all'\% & `RS_choice_last10'\% \\" _n
file write texfile "        DS-Contain & `DS_contain_all'\% & `DS_contain_last10'\% \\" _n
file write texfile "        \hline" _n
file write texfile "        \end{tabular}" _n
file write texfile "    \end{center}" _n
file write texfile "\end{table}" _n
file close texfile

display ""
display "Table 2 saved to: $mytables/table2.tex"


*------------------------------------------------------------------------------
// TABLE 3: Random effects probit regressions on preventive wars
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE 3: RANDOM EFFECTS PROBIT REGRESSIONS"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
gen aRSContain = (treatment=="2RS-Contain")
gen aRSCommit = (treatment=="3RS-Commit")
replace contain = 0 if contain == .
replace commit = 0 if commit == .
gen isCommit = (allo1b + commit > 11)

display "--- Model 1: All periods, treatment dummies ---"
xtprobit reject1 aRSContain aRSCommit if playerB==1 & treatment!="5DS-Contain" & treatment!="4RS-Choice", vce(cluster mgroup)
est store m1
margins, predict(pu0) dydx(*) post
est store m1_margins
test aRSContain = aRSCommit

display ""
display "--- Model 2: All periods, RS-Contain with actual choice ---"
xtprobit reject1 aRSContain contain if playerB==1 & (treatment=="1PreventiveWar" | treatment=="2RS-Contain"), vce(cluster mgroup)
est store m2
margins, predict(pu0) dydx(*) post
est store m2_margins

display ""
display "--- Model 3: All periods, RS-Commit with actual commitment ---"
xtprobit reject1 aRSCommit isCommit if playerB==1 & (treatment=="1PreventiveWar" | treatment=="3RS-Commit"), vce(cluster mgroup)
est store m3
margins, predict(pu0) dydx(*) post
est store m3_margins

display ""
display "--- Model 4: Last 10 periods, treatment dummies ---"
xtprobit reject1 aRSContain aRSCommit if playerB==1 & treatment!="5DS-Contain" & treatment!="4RS-Choice" & period>20, vce(cluster mgroup)
est store m4
margins, predict(pu0) dydx(*) post
est store m4_margins
test aRSContain = aRSCommit

display ""
display "--- Model 5: Last 10 periods, RS-Contain with actual choice ---"
xtprobit reject1 aRSContain contain if playerB==1 & (treatment=="1PreventiveWar" | treatment=="2RS-Contain") & period>20, vce(cluster mgroup)
est store m5
margins, predict(pu0) dydx(*) post
est store m5_margins

display ""
display "--- Model 6: Last 10 periods, RS-Commit with actual commitment ---"
xtprobit reject1 aRSCommit isCommit if playerB==1 & (treatment=="1PreventiveWar" | treatment=="3RS-Commit") & period>20, vce(cluster mgroup)
est store m6
margins, predict(pu0) dydx(*) post
est store m6_margins

* Export marginal effects
esttab m1_margins m2_margins m3_margins m4_margins m5_margins m6_margins using "$mytables/table3.tex", replace ///
	b(3) se(3) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize)

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// FIGURE 2: Policy choices over time
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE 2: Policy choices over time..."

use $mydata, clear
keep if treatment == "2RS-Contain"
gen choice1=(contain==1)
collapse y1 = choice1 , by(period)
twoway (connect y1 period, m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Frequency", size(`size')) title("RS-Contain")  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(0.1)1, labsize(`size') angle(0)) ///
		legend(on lab(1 "Containment") size(`size')  ///
		col(1) ring(0) pos(5) region(lcolor(none) fcolor(none))) scheme(s1color)
graph save "$data_folder/temp/1.gph", replace

use $mydata, clear
keep if treatment == "3RS-Commit"
gen total = allo1b+commit
gen choice1=(total>11)
collapse y1 = choice1 , by(period)
twoway (connect y1 period, m(Th) lpattern(dash) lcolor(black) mcolor(black))	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Frequency", size(`size')) title("RS-Commit")  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(0.1)1, labsize(`size') angle(0)) ///
		legend(on lab(1 "Commitment") size(`size')  ///
		col(1) ring(0) pos(5) region(lcolor(none) fcolor(none))) scheme(s1color)
graph save "$data_folder/temp/2.gph", replace

use $mydata, clear
keep if treatment == "4RS-Choice"
gen choice1=(choice==1)
gen choice2=(choice==2 & allo1b+commit>11)
collapse y1 = choice1 y2 = choice2 , by(period)
twoway (connect y1 period, m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y2 period, m(Th) lpattern(dash) lcolor(black) mcolor(black))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Frequency", size(`size')) title("RS-Choice") ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(0.1)1, labsize(`size') angle(0)) ///
		legend(lab(1 "Containment") lab(2 "Commitment")  size(`size')  ///
		col(1) ring(0) pos(1) region(lcolor(none) fcolor(none))) scheme(s1color)
graph save "$data_folder/temp/3.gph", replace

use $mydata, clear
keep if treatment == "5DS-Contain"
gen contain1 = (contain==1)
gen rival_contain1 = (rival_contain==1)
collapse y1 = contain1 y2 = rival_contain1 , by(period)
twoway (connect y1 period, m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y2 period, m(Ch) lpattern(solid) lcolor(black) mcolor(black))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Frequency", size(`size')) title("DS-Contain")  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(0.1)1, labsize(`size') angle(0)) ///
		legend(lab(1 "Self-contain") lab(2 "Contain by DS") size(`size')  ///
		col(1) ring(0) pos(1) region(lcolor(none) fcolor(none))) scheme(s1color)
graph save "$data_folder/temp/4.gph", replace

graph combine "$data_folder/temp/1.gph" "$data_folder/temp/2.gph"  ///
"$data_folder/temp/3.gph" "$data_folder/temp/4.gph" , ///
	scheme(s1mono) cols(2) xsize(2.5) ysize(2) scale(0.7)

graph export "$myfigs/figure2.png", replace as(png)


*------------------------------------------------------------------------------
// FIGURE 3: The frequency of preventive wars by policy choices
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE 3: The frequency of preventive wars..."

use $mydata, clear
gen isContain = 1 if contain == 1 & treatment == "2RS-Contain"
gen isCommit = 1 if treatment == "3RS-Commit" & allo1b + commit>11
replace isCommit = 0 if treatment == "3RS-Commit" & allo1b + commit==11
replace isContain = 1 if choice == 1 & treatment == "4RS-Choice"
replace isCommit = 1 if choice == 2 & allo1b + commit>11 & treatment == "4RS-Choice"
replace isCommit = 0 if choice == 2 & allo1b + commit<=11 & treatment == "4RS-Choice"
replace isContain = 1 if contain == 1 & treatment == "5DS-Contain"
gen isRivalContain = 1 if rival_contain == 1 & treatment == "5DS-Contain"
collapse y = reject1, by(treatment mgroup isContain isCommit isRivalContain)
collapse y = y (semean) b = y, by(treatment isContain isCommit isRivalContain)
replace y = y*100
replace b = b*100

gen y_u = y + b
gen y_l = y - b

gen mean_string = string(y, "%2.1f")

gen x = 1 if treatment == "1PreventiveWar"
replace x = 3 if treatment == "2RS-Contain" & isContain == .
replace x = 4 if treatment == "2RS-Contain" & isContain == 1
replace x = 6 if treatment == "3RS-Commit" & isCommit == .
replace x = 7 if treatment == "3RS-Commit" & isCommit == 1
replace x = 9 if treatment == "4RS-Choice" & isCommit == .
replace x = 10 if treatment == "4RS-Choice" & isContain == 1
replace x = 11 if treatment == "4RS-Choice" & isCommit == 1
replace x = 13 if treatment == "5DS-Contain" & isContain == . & isRivalContain == .
replace x = 14 if treatment == "5DS-Contain" & isContain == 1
replace x = 15 if treatment == "5DS-Contain" & isRivalContain == 1

twoway (bar y x if isContain == . & isCommit == . , bcol(gs1)) ///
	(bar y x if isContain == 1, bcol(gs10)) ///
	(bar y x if isCommit == 1, fcol(gs16) lcolor(gs1) lw(medium)) ///
	(bar y x if isRivalContain == 1, bcol(gs14)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(mediumlarge) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Fighting Rate (%)", size(`size'))  ///
	xlabel(1 "PW-Baseline" 3.5 "RS-Contain" 6.5 "RS-Commit" 10 "RS-Choice" 14 "DS-Contain", labgap(5) labsize(`size') notick ) ///
	ylabel(0(20)100) legend(lab(1 "No action") lab(2 "Self-contain") lab(3 "Commit") lab(4 "Contain by DS") order(1 2 3 4) size(`size') ///
		col(4) ring(1) pos(6) region(lcolor(none) fcolor(none))) scheme(s1mono) xsize(2) ysize(1)

graph export "$myfigs/figure3.png", replace as(png)


*------------------------------------------------------------------------------
// FIGURE 4: The rising state's and declining state's average payoffs
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE 4: Average payoffs by treatment..."

use $mydata, clear
keep if playerB == 0
replace profit = profit - 5

gen isContain = 1 if contain == 1 & treatment == "2RS-Contain"
gen isCommit = 1 if treatment == "3RS-Commit" & allo1b + commit>11
replace isCommit = 0 if treatment == "3RS-Commit" & allo1b + commit==11
replace isContain = 1 if choice == 1 & treatment == "4RS-Choice"
replace isCommit = 1 if choice == 2 & allo1b + commit>11 & treatment == "4RS-Choice"
replace isCommit = 0 if choice == 2 & allo1b + commit<=11 & treatment == "4RS-Choice"
replace isContain = 1 if contain == 1 & treatment == "5DS-Contain"
gen isRivalContain = 1 if rival_contain == 1 & treatment == "5DS-Contain"
collapse y = profit, by(treatment mgroup isContain isCommit isRivalContain)
collapse y = y (semean) b = y, by(treatment isContain isCommit isRivalContain)
gen y_u = y + b
gen y_l = y - b

gen mean_string = string(y, "%2.1f")

gen x = 1 if treatment == "1PreventiveWar"
replace x = 3 if treatment == "2RS-Contain" & isContain == .
replace x = 4 if treatment == "2RS-Contain" & isContain == 1
replace x = 6 if treatment == "3RS-Commit" & isCommit == .
replace x = 7 if treatment == "3RS-Commit" & isCommit == 1
replace x = 9 if treatment == "4RS-Choice" & isContain == . & isCommit == .
replace x = 10 if treatment == "4RS-Choice" & isContain == 1
replace x = 11 if treatment == "4RS-Choice" & isCommit == 1
replace x = 13 if treatment == "5DS-Contain" & isContain == . & isRivalContain == .
replace x = 14 if treatment == "5DS-Contain" & isContain == 1
replace x = 15 if treatment == "5DS-Contain" & isRivalContain == 1

twoway (bar y x if isContain == . & isCommit == . , bcol(gs1)) ///
	(bar y x if isContain == 1, bcol(gs10)) ///
	(bar y x if isCommit == 1, fcol(gs16) lcolor(gs1) lw(medium)) ///
	(bar y x if isRivalContain == 1, bcol(gs14)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(mediumlarge) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Average Payoff", size(`size')) title("Rising State") ///
	xlabel(1 "PW-Baseline" 3.5 "RS-Contain" 6.5 "RS-Commit" 10 "RS-Choice" 14 "DS-Contain", labsize(`size') notick ) ///
	ylabel(0(4)16) legend(lab(1 "No action") lab(2 "Self-contain") lab(3 "Commit") lab(4 "Contain by DS") order(1 2 3 4) size(`size') ///
		col(4) ring(1) pos(6) region(lcolor(none) fcolor(none))) scheme(s1mono) xsize(2) ysize(1)

graph save "$data_folder/temp/1.gph", replace

use $mydata, clear
keep if playerB == 1
replace profit = profit - 5
gen isContain = 1 if contain == 1 & treatment == "2RS-Contain"
gen isCommit = 1 if treatment == "3RS-Commit" & allo1b + commit>11
replace isCommit = 0 if treatment == "3RS-Commit" & allo1b + commit==11
replace isContain = 1 if choice == 1 & treatment == "4RS-Choice"
replace isCommit = 1 if choice == 2 & allo1b + commit>11 & treatment == "4RS-Choice"
replace isCommit = 0 if choice == 2 & allo1b + commit<=11 & treatment == "4RS-Choice"
replace isContain = 1 if contain == 1 & treatment == "5DS-Contain"
gen isRivalContain = 1 if rival_contain == 1 & treatment == "5DS-Contain"
collapse y = profit, by(treatment mgroup isContain isCommit isRivalContain)
collapse y = y (semean) b = y, by(treatment isContain isCommit isRivalContain)
gen y_u = y + b
gen y_l = y - b

gen mean_string = string(y, "%2.1f")

gen x = 1 if treatment == "1PreventiveWar"
replace x = 3 if treatment == "2RS-Contain" & isContain == .
replace x = 4 if treatment == "2RS-Contain" & isContain == 1
replace x = 6 if treatment == "3RS-Commit" & isCommit == .
replace x = 7 if treatment == "3RS-Commit" & isCommit == 1
replace x = 9 if treatment == "4RS-Choice" & isContain == . & isCommit == .
replace x = 10 if treatment == "4RS-Choice" & isContain == 1
replace x = 11 if treatment == "4RS-Choice" & isCommit == 1
replace x = 13 if treatment == "5DS-Contain" & isContain == . & isRivalContain == .
replace x = 14 if treatment == "5DS-Contain" & isContain == 1
replace x = 15 if treatment == "5DS-Contain" & isRivalContain == 1

twoway (bar y x if isContain == . & isCommit == . , bcol(gs1)) ///
	(bar y x if isContain == 1, bcol(gs10)) ///
	(bar y x if isCommit == 1, fcol(gs16) lcolor(gs1) lw(medium)) ///
	(bar y x if isRivalContain == 1, bcol(gs14)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(mediumlarge) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Average Payoff", size(`size')) title("Declining State") ///
	xlabel(1 "PW-Baseline" 3.5 "RS-Contain" 6.5 "RS-Commit" 10 "RS-Choice" 14 "DS-Contain", labsize(`size') notick ) ///
	ylabel(0(4)16) legend(lab(1 "No action") lab(2 "Self-contain") lab(3 "Commit") lab(4 "Contain by DS") order(1 2 3 4) size(`size') ///
		col(4) ring(1) pos(6) region(lcolor(none) fcolor(none))) scheme(s1mono) xsize(2) ysize(1)

		graph save "$data_folder/temp/2.gph", replace

grc1leg2 "$data_folder/temp/1.gph" "$data_folder/temp/2.gph" , ///
	scheme(s1mono) cols(1) lcols(4) pos(6)  ytol1title xsize(2) ysize(2) scale(0.75)

graph export "$myfigs/figure4.png", replace as(png)


*------------------------------------------------------------------------------
// TABLE 4: Random effects probit regressions in RS-Choice
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE 4: RS-CHOICE REGRESSION"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
gen aRSChoice = (treatment=="4RS-Choice")
gen isContain = (choice == 1 & treatment == "4RS-Choice")
gen isCommit = (choice == 2 & allo1b + commit > 11 & treatment == "4RS-Choice")
keep if treatment=="1PreventiveWar" | treatment=="4RS-Choice"

display "--- Model 1: All periods, treatment dummy ---"
xtprobit reject1 aRSChoice if playerB==1, vce(cluster mgroup)
est store t4m1
margins, predict(pu0) dydx(*) post
est store t4m1_margins

display ""
display "--- Model 2: All periods, with actual choices ---"
xtprobit reject1 aRSChoice isContain isCommit if playerB==1, vce(cluster mgroup)
est store t4m2
margins, predict(pu0) dydx(*) post
est store t4m2_margins
test isContain == isCommit

display ""
display "--- Model 3: Last 10 periods, treatment dummy ---"
xtprobit reject1 aRSChoice if playerB==1 & period>20, vce(cluster mgroup)
est store t4m3
margins, predict(pu0) dydx(*) post
est store t4m3_margins

display ""
display "--- Model 4: Last 10 periods, with actual choices ---"
xtprobit reject1 aRSChoice isContain isCommit if playerB==1 & period>20, vce(cluster mgroup)
est store t4m4
margins, predict(pu0) dydx(*) post
est store t4m4_margins
test isContain == isCommit

* Export marginal effects
esttab t4m1_margins t4m2_margins t4m3_margins t4m4_margins using "$mytables/table4.tex", replace ///
	b(3) se(3) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize)

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// TABLE 5: Random effects probit regressions in DS-Contain
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE 5: DS-CONTAIN REGRESSION"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
keep if treatment=="2RS-Contain" | treatment=="5DS-Contain"
gen aDSContain = (treatment=="5DS-Contain")
gen is_contain = (contain==1)
gen is_rival_contain = (rival_contain==1)
gen term = aDSContain * is_contain

display "--- Model 1: All periods, treatment dummy ---"
xtprobit reject1 aDSContain if playerB==1, vce(cluster mgroup)
est store t5m1
margins, predict(pu0) dydx(*) post
est store t5m1_margins

display ""
display "--- Model 2: All periods, with actual choices ---"
xtprobit reject1 aDSContain is_contain term is_rival_contain if playerB==1, vce(cluster mgroup)
est store t5m2
margins, predict(pu0) dydx(*) post
est store t5m2_margins

display ""
display "--- Model 3: Last 10 periods, treatment dummy ---"
xtprobit reject1 aDSContain if playerB==1 & period>20, vce(cluster mgroup)
est store t5m3
margins, predict(pu0) dydx(*) post
est store t5m3_margins

display ""
display "--- Model 4: Last 10 periods, with actual choices ---"
xtprobit reject1 aDSContain is_contain term is_rival_contain if playerB==1 & period>20, vce(cluster mgroup)
est store t5m4
margins, predict(pu0) dydx(*) post
est store t5m4_margins

* Export marginal effects
esttab t5m1_margins t5m2_margins t5m3_margins t5m4_margins using "$mytables/table5.tex", replace ///
	b(3) se(3) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize)

est clear

display ""
display "========================================================"


//==============================================================================
// APPENDIX - FIGURES AND TABLES
//==============================================================================


*------------------------------------------------------------------------------
// TABLE C1: Conditional on Stage 1 offer in RS-Contain
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE C1: CONDITIONAL ON STAGE 1 OFFER (RS-CONTAIN)"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
gen aRSContain = (treatment=="2RS-Contain")
replace contain = 0 if contain == .
gen allo1b_high =(allo1b == 10)
gen high_contain = allo1b_high * contain
keep if treatment=="1PreventiveWar" | treatment=="2RS-Contain"

display "--- All periods ---"
xtprobit reject1 aRSContain contain allo1b_high high_contain if playerB==1, vce(cluster mgroup)
est store tc1m1
margins, predict(pu0) dydx(*) post
est store tc1m1_margins

display ""
display "--- Last 10 periods ---"
xtprobit reject1 aRSContain contain allo1b_high high_contain if playerB==1 & period>20, vce(cluster mgroup)
est store tc1m2
margins, predict(pu0) dydx(*) post
est store tc1m2_margins

* Export marginal effects
esttab tc1m1_margins tc1m2_margins using "$mytables/tablec1.tex", replace ///
	b(3) se(3) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize)

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// TABLE C2: Payoff regressions in RS-Contain
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE C2: PAYOFF REGRESSIONS (RS-CONTAIN)"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
keep if treatment=="1PreventiveWar" | treatment=="2RS-Contain"
gen aRSContain = (treatment=="2RS-Contain")
replace contain = 0 if contain == .
xtreg profit aRSContain##contain##playerB, cluster(mgroup)
est store tc2m1
test 1.aRSContain + 1.aRSContain#1.playerB = 0
test 1.contain + 1.contain#1.playerB = 0
xtreg profit aRSContain##contain##playerB if period>20, cluster(mgroup)
est store tc2m2
test 1.aRSContain + 1.aRSContain#1.playerB = 0
test 1.contain + 1.contain#1.playerB = 0

* Export models
esttab tc2m1 tc2m2 using "$mytables/tablec2.tex", replace ///
	b(3) se(3) scalar(N r2_a) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize) 

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// TABLE C3: Payoff regressions in RS-Commit
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE C3: PAYOFF REGRESSIONS (RS-COMMIT)"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
keep if treatment=="1PreventiveWar" | treatment=="3RS-Commit"
gen aRSCommit = (treatment=="3RS-Commit")
replace commit = 0 if commit == .
gen isCommit = (allo1b + commit > 11)
xtreg profit aRSCommit##isCommit##playerB, cluster(mgroup)
est store tc3m1
test 1.aRSCommit + 1.aRSCommit#1.playerB = 0
test 1.isCommit + 1.isCommit#1.playerB = 0
xtreg profit aRSCommit##isCommit##playerB if period>20, cluster(mgroup)
est store tc3m2
test 1.aRSCommit + 1.aRSCommit#1.playerB = 0
test 1.isCommit + 1.isCommit#1.playerB = 0

* Export models
esttab tc3m1 tc3m2 using "$mytables/tablec3.tex", replace ///
	b(3) se(3) scalar(N r2_a) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize) 

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// TABLE C4: Conditional on Stage 1 offer in RS-Choice
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE C4: CONDITIONAL ON STAGE 1 OFFER (RS-CHOICE)"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
gen aRSChoice = (treatment=="4RS-Choice")
gen isCommit = (choice==2)
drop if treatment=="4RS-Choice" & choice==1
replace allo1b = allo1b + commit if treatment=="4RS-Choice"
gen allo1b_high =(allo1b > 11)
keep if treatment=="1PreventiveWar" | treatment=="4RS-Choice"

display "--- All periods ---"
xtprobit reject1 aRSChoice isCommit allo1b_high if playerB==1, vce(cluster mgroup)
est store tc4m1
margins, predict(pu0) dydx(*) post
est store tc4m1_margins

display ""
display "--- Last 10 periods ---"
xtprobit reject1 aRSChoice isCommit allo1b_high if playerB==1 & period>20, vce(cluster mgroup)
est store tc4m2
margins, predict(pu0) dydx(*) post
est store tc4m2_margins

* Export marginal effects
esttab tc4m1_margins tc4m2_margins using "$mytables/tablec4.tex", replace ///
	b(3) se(3) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize)

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// TABLE C5: Payoff regressions in RS-Choice
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE C5: PAYOFF REGRESSIONS (RS-CHOICE)"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
gen aRSChoice = (treatment=="4RS-Choice")
keep if treatment=="1PreventiveWar" | treatment=="4RS-Choice"
gen isContain = (choice==1)
gen isCommit = (choice==2 & allo1b + commit>11)
xtreg profit aRSChoice##isContain##playerB aRSChoice##isCommit##playerB, cluster(mgroup)
est store tc5m1
test 1.aRSChoice + 1.aRSChoice#1.playerB = 0
test 1.isContain + 1.isContain#1.playerB = 0
test 1.isCommit + 1.isCommit#1.playerB = 0
xtreg profit aRSChoice##isContain##playerB aRSChoice##isCommit##playerB if period>20, cluster(mgroup)
est store tc5m2
test 1.aRSChoice + 1.aRSChoice#1.playerB = 0
test 1.isContain + 1.isContain#1.playerB = 0
test 1.isCommit + 1.isCommit#1.playerB = 0

* Export models
esttab tc5m1 tc5m2 using "$mytables/tablec5.tex", replace ///
	b(3) se(3) scalar(N r2_a) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize) 

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// TABLE C6: Payoff regressions in DS-Contain
*------------------------------------------------------------------------------

display ""
display "========================================================"
display "TABLE C6: PAYOFF REGRESSIONS (DS-CONTAIN)"
display "========================================================"
display ""

use $mydata, clear
xtset pid period
keep if treatment=="1PreventiveWar" | treatment=="5DS-Contain"
gen aDSContain = (treatment=="5DS-Contain")
gen is_contain = (contain==1)
gen is_rival_contain = (rival_contain==1)
xtreg profit aDSContain##is_contain##playerB aDSContain##is_rival_contain##playerB, cluster(mgroup)
est store tc6m1
test 1.aDSContain + 1.aDSContain#1.playerB = 0
test 1.is_contain + 1.is_contain#1.playerB = 0
test 1.is_rival_contain + 1.is_rival_contain#1.playerB = 0
xtreg profit aDSContain##is_contain##playerB aDSContain##is_rival_contain##playerB if period>20, cluster(mgroup)
est store tc6m2
test 1.aDSContain + 1.aDSContain#1.playerB = 0
test 1.is_contain + 1.is_contain#1.playerB = 0
test 1.is_rival_contain + 1.is_rival_contain#1.playerB = 0

* Export models
esttab tc6m1 tc6m2 using "$mytables/tablec6.tex", replace ///
	b(3) se(3) scalar(N r2_a) ///
	star(* 0.1 ** 0.05 *** 0.01) ///
	nolines nonumbers nomtitles compress ///
	booktabs fragment ///
	label width(0.6\hsize) 

est clear

display ""
display "========================================================"


*------------------------------------------------------------------------------
// APPENDIX FIGURE C2: REJECTION 1 over time in RS-Contain
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C2: Rejection 1 over time in RS-Contain..."

use $mydata, clear
collapse y = reject1, by(period treatment contain)
replace y = y*100

twoway (connect y period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y period if treatment == "2RS-Contain" & contain==1, m(Th) lpattern(dash) lcolor(green) mcolor(green) lwidth(medium) )	///
		(connect y period if treatment == "2RS-Contain" & contain==0, m(Dh) lpattern(dash) lcolor(red) mcolor(red) lwidth(medium))	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Fighting Rate (%)", size(`size'))  ///
		xlabel(2(2)30, labsize(`size') ) ylabel(0(20)100, labsize(`size') angle(0)) ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Contain, contain") lab(3 "RS-Contain, no action") size(`size')  ///
		col(3) pos(6) region(lcolor(none) fcolor(none))) scheme(s1color) scale(0.9)

graph export "$myfigs/figurec2.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C3: REJECTION 1 over time in RS-Commit
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C3: Rejection 1 over time in RS-Commit..."

use $mydata, clear
gen total = 1 if allo1b + commit>11
replace total = 0 if allo1b + commit==11
collapse y = reject1, by(period treatment total)
replace y = y*100

twoway (connect y period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y period if treatment == "3RS-Commit" & total==1, m(Th) lpattern(dash) lcolor(blue) mcolor(blue) lwidth(medium) )	///
		(connect y period if treatment == "3RS-Commit" & total==., m(Dh) lpattern(dash) lcolor(red) mcolor(red) lwidth(medium) )	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Fighting Rate (%)", size(`size'))  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(20)100, labsize(`size') angle(0)) ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Commit, commit") lab(3 "RS-Commit, no action") size(`size')  ///
		col(3) pos(6) region(lcolor(none) fcolor(none))) scheme(s1color) scale(0.9)

graph export "$myfigs/figurec3.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C4: Preventive war rate in last 10 rounds
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C4: Preventive war rate in last 10 rounds..."

use $mydata, clear
keep if period>20
gen isContain = 1 if contain == 1 & treatment == "2RS-Contain"
gen isCommit = 1 if treatment == "3RS-Commit" & allo1b + commit>11
replace isCommit = 0 if treatment == "3RS-Commit" & allo1b + commit==11
replace isContain = 1 if choice == 1 & treatment == "4RS-Choice"
replace isCommit = 1 if choice == 2 & allo1b + commit>11 & treatment == "4RS-Choice"
replace isCommit = 0 if choice == 2 & allo1b + commit<=11 & treatment == "4RS-Choice"
replace isContain = 1 if contain == 1 & treatment == "5DS-Contain"
gen isRivalContain = 1 if rival_contain == 1 & treatment == "5DS-Contain"
collapse y = reject1, by(treatment mgroup isContain isCommit isRivalContain)
collapse y = y (semean) b = y, by(treatment isContain isCommit isRivalContain)
replace y = y*100
replace b = b*100

gen y_u = y + b
gen y_l = y - b

gen mean_string = string(y, "%2.1f")

gen x = 1 if treatment == "1PreventiveWar"
replace x = 3 if treatment == "2RS-Contain" & isContain == .
replace x = 4 if treatment == "2RS-Contain" & isContain == 1
replace x = 6 if treatment == "3RS-Commit" & isCommit == .
replace x = 7 if treatment == "3RS-Commit" & isCommit == 1
replace x = 9 if treatment == "4RS-Choice" & isContain == . & isCommit == .
replace x = 10 if treatment == "4RS-Choice" & isContain == 1
replace x = 11 if treatment == "4RS-Choice" & isCommit == 1
replace x = 13 if treatment == "5DS-Contain" & isContain == . & isRivalContain == .
replace x = 14 if treatment == "5DS-Contain" & isContain == 1
replace x = 15 if treatment == "5DS-Contain" & isRivalContain == 1

twoway (bar y x if isContain == . & isCommit == . , bcol(gs1)) ///
	(bar y x if isContain == 1, bcol(gs10)) ///
	(bar y x if isCommit == 1, fcol(gs16) lcolor(gs1) lw(medium)) ///
	(bar y x if isRivalContain == 1, bcol(gs14)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(mediumlarge) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Fighting Rate (%)", size(`size'))  ///
	xlabel(1 "PW-Baseline" 3.5 "RS-Contain" 6.5 "RS-Commit" 10 "RS-Choice" 14 "DS-Contain", labgap(5) labsize(`size') notick ) ///
	ylabel(0(20)100) legend(lab(1 "No action") lab(2 "Self-contain") lab(3 "Commit") lab(4 "Contain by DS") order(1 2 3 4) size(`size') ///
		col(4) ring(1) pos(6) region(lcolor(none) fcolor(none))) scheme(s1mono) xsize(2) ysize(1)

graph export "$myfigs/figurec4.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C5: Offers over time in RS-Contain
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C5: Offers over time in RS-Contain..."

use $mydata, clear
replace allo2b = . if reject1==1
collapse y1 = allo1b y2 = allo2b, by(period treatment contain)
twoway (connect y1 period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y1 period if treatment == "2RS-Contain" & contain==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	////
		(connect y1 period if treatment == "2RS-Contain" & contain==0, m(Dh) lpattern(longdash) lcolor(red) mcolor(red))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)10, labsize(`size') angle(0)) ///
		title("Stage 1") ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Contain, contain") lab(3 "RS-Contain, no action") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/1.gph", replace
twoway 	(connect y2 period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y2 period if treatment == "2RS-Contain" & contain==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	////
		(connect y2 period if treatment == "2RS-Contain" & contain==0, m(Dh) lpattern(longdash) lcolor(red) mcolor(red))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		title("Stage 2") ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)10, labsize(`size') angle(0)) ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Contain, contain") lab(3 "RS-Contain, no action") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/2.gph", replace
grc1leg2 "$data_folder/temp/1.gph" "$data_folder/temp/2.gph" , ///
	scheme(s1color) cols(2) lcols(4) pos(6) labsize(small) ytol1title xsize(2) ysize(1) scale(1.3)

graph export "$myfigs/figurec5.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C6: Conditional rejection in RS-Contain
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C6: Conditional rejection in RS-Contain..."

use $mydata, clear
gen count = 1
keep if treatment == "2RS-Contain" | treatment == "1PreventiveWar"
collapse (count) count, by(treatment contain allo1b reject1)

bys contain allo1b: egen a1 = total(count)
gen rejFreq = count/a1 if reject1 == 1
bys contain: egen x = total(count)
keep if reject1 == 1
gen alloFreq = a1/x

gen contain1 = "PW-Baseline" if treatment == "1PreventiveWar"
replace contain1 = "RS-Contain, no action" if contain==0
replace contain1 = "RS-Contain, contain" if contain==1

twoway (bar alloFreq allo1b) ///
	(connect rejFreq allo1b, lpattern(solid) col(black)), ///
	by(contain1, note("") col(3)) ///
	xlabel(0(2)10) ylabel(0(.2)1) ///
	xtitle("Rising state's offer in stage 1") ytitle("Frequency")  ///
	legend(lab(1 "Frequency of stage 1 offer") lab(2 "Frequency of fighting (preventive war)")  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none))) scheme(s1mono)

graph export "$myfigs/figurec6.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C7: Offers over time in RS-Commit
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C7: Offers over time in RS-Commit..."

use $mydata, clear
replace allo2b = . if reject1==1
replace allo1b = allo1b + commit if treatment == "3RS-Commit"
collapse y1 = allo1b y2 = allo2b, by(period treatment contain)
twoway (connect y1 period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y1 period if treatment == "3RS-Commit", m(Th) lpattern(dash) lcolor(blue) mcolor(blue))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)14, labsize(`size') angle(0)) ///
		title("Stage 1 (including advance allocation)") ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Commit") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/1.gph", replace
twoway 	(connect y2 period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y2 period if treatment == "3RS-Commit", m(Th) lpattern(dash) lcolor(blue) mcolor(blue))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		title("Stage 2") ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)10, labsize(`size') angle(0)) ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Commit") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/2.gph", replace
grc1leg2 "$data_folder/temp/1.gph" "$data_folder/temp/2.gph" , ///
	scheme(s1color) cols(2) lcols(4) pos(6) labsize(small) ytol1title xsize(2) ysize(1) scale(1.3)

graph export "$myfigs/figurec7.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C8: Conditional rejection in RS-Commit
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C8: Conditional rejection in RS-Commit..."

use $mydata, clear
gen count = 1
keep if treatment == "3RS-Commit"
gen total = allo1b + commit
collapse (count) count, by(total reject1)

bys total: egen a1 = total(count)
gen rejFreq = count/a1 if reject1 == 1
egen x = total(count)
keep if reject1 == 1
gen alloFreq = a1/x

twoway (bar alloFreq total) ///
	(connect rejFreq total, lpattern(solid) col(black)), ///
	xlabel(0(2)20) ylabel(0(.2)1) ///
	xtitle("Rising state's offer in stage 1") ytitle("Frequency")  ///
	legend(lab(1 "Frequency of stage 1 offer (including advance allocation)") lab(2 "Frequency of fighting (preventive war)")  ///
		col(1)  region(lcolor(none) fcolor(none))) scheme(s1mono)

graph export "$myfigs/figurec8.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C9: Payoffs in last 10 rounds
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C9: Payoffs in last 10 rounds..."

use $mydata, clear
keep if period>20
keep if playerB == 0
replace profit = profit - 5

gen isContain = 1 if contain == 1 & treatment == "2RS-Contain"
gen isCommit = 1 if treatment == "3RS-Commit" & allo1b + commit>11
replace isCommit = 0 if treatment == "3RS-Commit" & allo1b + commit==11
replace isContain = 1 if choice == 1 & treatment == "4RS-Choice"
replace isCommit = 1 if choice == 2 & allo1b + commit>11 & treatment == "4RS-Choice"
replace isCommit = 0 if choice == 2 & allo1b + commit<=11 & treatment == "4RS-Choice"
replace isContain = 1 if contain == 1 & treatment == "5DS-Contain"
gen isRivalContain = 1 if rival_contain == 1 & treatment == "5DS-Contain"
collapse y = profit, by(treatment mgroup isContain isCommit isRivalContain)
collapse y = y (semean) b = y, by(treatment isContain isCommit isRivalContain)
gen y_u = y + b
gen y_l = y - b

gen mean_string = string(y, "%2.1f")

gen x = 1 if treatment == "1PreventiveWar"
replace x = 3 if treatment == "2RS-Contain" & isContain == .
replace x = 4 if treatment == "2RS-Contain" & isContain == 1
replace x = 6 if treatment == "3RS-Commit" & isCommit == .
replace x = 7 if treatment == "3RS-Commit" & isCommit == 1
replace x = 9 if treatment == "4RS-Choice" & isContain == . & isCommit == .
replace x = 10 if treatment == "4RS-Choice" & isContain == 1
replace x = 11 if treatment == "4RS-Choice" & isCommit == 1
replace x = 13 if treatment == "5DS-Contain" & isContain == . & isRivalContain == .
replace x = 14 if treatment == "5DS-Contain" & isContain == 1
replace x = 15 if treatment == "5DS-Contain" & isRivalContain == 1

twoway (bar y x if isContain == . & isCommit == . , bcol(gs1)) ///
	(bar y x if isContain == 1, bcol(gs10)) ///
	(bar y x if isCommit == 1, fcol(gs16) lcolor(gs1) lw(medium)) ///
	(bar y x if isRivalContain == 1, bcol(gs14)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(mediumlarge) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Average Payoff", size(`size')) title("Rising State") ///
	xlabel(1 "PW-Baseline" 3.5 "RS-Contain" 6.5 "RS-Commit" 10 "RS-Choice" 14 "DS-Contain", labsize(`size') notick ) ///
	ylabel(0(4)16) legend(lab(1 "No action") lab(2 "Self-contain") lab(3 "Commit") lab(4 "Contain by DS") order(1 2 3 4) size(`size') ///
		col(4) ring(1) pos(6) region(lcolor(none) fcolor(none))) scheme(s1mono) xsize(2) ysize(1)

graph save "$data_folder/temp/1.gph", replace

use $mydata, clear
keep if period>20
keep if playerB == 1
replace profit = profit - 5
gen isContain = 1 if contain == 1 & treatment == "2RS-Contain"
gen isCommit = 1 if treatment == "3RS-Commit" & allo1b + commit>11
replace isCommit = 0 if treatment == "3RS-Commit" & allo1b + commit==11
replace isContain = 1 if choice == 1 & treatment == "4RS-Choice"
replace isCommit = 1 if choice == 2 & allo1b + commit>11 & treatment == "4RS-Choice"
replace isCommit = 0 if choice == 2 & allo1b + commit<=11 & treatment == "4RS-Choice"
replace isContain = 1 if contain == 1 & treatment == "5DS-Contain"
gen isRivalContain = 1 if rival_contain == 1 & treatment == "5DS-Contain"
collapse y = profit, by(treatment mgroup isContain isCommit isRivalContain)
collapse y = y (semean) b = y, by(treatment isContain isCommit isRivalContain)
gen y_u = y + b
gen y_l = y - b

gen mean_string = string(y, "%2.1f")

gen x = 1 if treatment == "1PreventiveWar"
replace x = 3 if treatment == "2RS-Contain" & isContain == .
replace x = 4 if treatment == "2RS-Contain" & isContain == 1
replace x = 6 if treatment == "3RS-Commit" & isCommit == .
replace x = 7 if treatment == "3RS-Commit" & isCommit == 1
replace x = 9 if treatment == "4RS-Choice" & isContain == . & isCommit == .
replace x = 10 if treatment == "4RS-Choice" & isContain == 1
replace x = 11 if treatment == "4RS-Choice" & isCommit == 1
replace x = 13 if treatment == "5DS-Contain" & isContain == . & isRivalContain == .
replace x = 14 if treatment == "5DS-Contain" & isContain == 1
replace x = 15 if treatment == "5DS-Contain" & isRivalContain == 1

twoway (bar y x if isContain == . & isCommit == . , bcol(gs1)) ///
	(bar y x if isContain == 1, bcol(gs10)) ///
	(bar y x if isCommit == 1, fcol(gs16) lcolor(gs1) lw(medium)) ///
	(bar y x if isRivalContain == 1, bcol(gs14)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(mediumlarge) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Average Payoff", size(`size')) title("Declining State") ///
	xlabel(1 "PW-Baseline" 3.5 "RS-Contain" 6.5 "RS-Commit" 10 "RS-Choice" 14 "DS-Contain", labsize(`size') notick ) ///
	ylabel(0(4)16) legend(lab(1 "No action") lab(2 "Self-contain") lab(3 "Commit") lab(4 "Contain by DS") order(1 2 3 4) size(`size') ///
		col(4) ring(1) pos(6) region(lcolor(none) fcolor(none))) scheme(s1mono) xsize(2) ysize(1)

		graph save "$data_folder/temp/2.gph", replace

grc1leg2 "$data_folder/temp/1.gph" "$data_folder/temp/2.gph" , ///
	scheme(s1mono) cols(1) lcols(4) pos(6)  ytol1title xsize(2) ysize(2) scale(0.75)

graph export "$myfigs/figurec9.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C10: REJECTION 1 over time in RS-Choice
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C10: Rejection 1 over time in RS-Choice..."

use $mydata, clear
gen total = 1 if allo1b + commit>11
replace total = 0 if allo1b + commit==11
collapse y = reject1, by(period treatment choice total)
replace y = y*100

twoway (connect y period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y period if treatment == "4RS-Choice" & choice==1, m(Th) lpattern(dash) lcolor(green) mcolor(green) lwidth(medium))	///
		(connect y period if treatment == "4RS-Choice" & choice==2 & total==1, m(Dh) lpattern(longdash) lcolor(blue) mcolor(blue))	///
		(connect y period if treatment == "4RS-Choice" & choice==3, m(Sh) lpattern(solid) lcolor(red) mcolor(red))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Fighting Rate (%)", size(`size'))  ///
		xlabel(2(2)30, labsize(`size') ) ylabel(0(20)100, labsize(`size') angle(0)) ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Choice, contain") lab(3 "RS-Choice, commit") lab(4 "RS-Choice, no action") size(`size')  ///
		col(3) pos(6) region(lcolor(none) fcolor(none))) scheme(s1color) scale(0.9)

graph export "$myfigs/figurec10.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C11: Offers over time in RS-Choice
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C11: Offers over time in RS-Choice..."

use $mydata, clear
replace allo2b = . if reject1==1
replace allo1b = allo1b + commit if treatment == "4RS-Choice" & choice==2
collapse y1 = allo1b y2 = allo2b, by(period treatment choice)
twoway (connect y1 period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y1 period if treatment == "4RS-Choice" & choice==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	////
		(connect y1 period if treatment == "4RS-Choice" & choice==2, m(Dh) lpattern(longdash) lcolor(blue) mcolor(blue))	////
		(connect y1 period if treatment == "4RS-Choice" & choice==3, m(Sh) lpattern(solid) lcolor(red) mcolor(red))	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)14, labsize(`size') angle(0)) ///
		title("Stage 1 (including advance allocation)") ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Choice, contain") lab(3 "RS-Choice, commit") lab(4 "RS-Choice, no action") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/1.gph", replace
twoway 	(connect y2 period if treatment == "1PreventiveWar", m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick) )	///
		(connect y2 period if treatment == "4RS-Choice" & choice==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	////
		(connect y2 period if treatment == "4RS-Choice" & choice==2, m(Dh) lpattern(longdash) lcolor(blue) mcolor(blue))	////
		(connect y2 period if treatment == "4RS-Choice" & choice==3, m(Sh) lpattern(solid) lcolor(red) mcolor(red))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		title("Stage 2") ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)10, labsize(`size') angle(0)) ///
		legend(lab(1 "PW-Baseline") lab(2 "RS-Choice, contain") lab(3 "RS-Choice, commit") lab(4 "RS-Choice, no action") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/2.gph", replace
grc1leg2 "$data_folder/temp/1.gph" "$data_folder/temp/2.gph" , ///
	scheme(s1color) cols(2) lcols(4) pos(6) labsize(small) ytol1title xsize(2) ysize(1) scale(1.3)

graph export "$myfigs/figurec11.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C12: Conditional rejection in RS-Choice
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C12: Conditional rejection in RS-Choice..."

use $mydata, clear
gen count = 1
keep if treatment=="4RS-Choice"
replace allo1b = allo1b + commit if choice==2
collapse (count) count, by(choice allo1b reject1)

bys choice allo1b: egen a1 = total(count)
gen rejFreq = count/a1 if reject1 == 1
bys choice: egen x = total(count)
keep if reject1 == 1
gen alloFreq = a1/x

gen choice1 = "No action" if choice==3
replace choice1 = "Contain" if choice==1
replace choice1 = "Commit" if choice==2

twoway (bar alloFreq allo1b) ///
	(connect rejFreq allo1b, lpattern(solid) col(black)), ///
	by(choice1, note("") col(3)) ///
	xlabel(0(2)16) ylabel(0(.2)1) ///
	xtitle("Rising state's offer in stage 1") ytitle("Frequency")  ///
	legend(lab(1 "Frequency of stage 1 offer (including advance allocation)") lab(2 "Frequency of fighting (preventive war)")  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none))) scheme(s1mono)

graph export "$myfigs/figurec12.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C13: REJECTION 1 over time in DS-Contain
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C13: Rejection 1 over time in DS-Contain..."

use $mydata, clear

collapse y = reject1, by(period treatment contain rival_contain)
replace y = y*100

twoway (connect y period if treatment == "5DS-Contain" & contain==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	////
		(connect y period if treatment == "5DS-Contain" & contain==0 & rival_contain==0, m(Sh) lpattern(solid) lcolor(red) mcolor(red))	////
		(connect y period if treatment == "5DS-Contain" & rival_contain==1, m(Dh) lpattern(longdash) lcolor(gs7) mcolor(gs7))	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Fighting Rate (%)", size(`size'))  ///
		xlabel(2(2)30, labsize(`size') ) ylabel(0(20)100, labsize(`size') angle(0)) ///
		legend(lab(1 "Self-contain") lab(2 "No action") lab(3 "Contain by DS") size(`size')  ///
		col(3) pos(6) region(lcolor(none) fcolor(none))) scheme(s1color) scale(0.9)

graph export "$myfigs/figurec13.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C14: Offers over time in DS-Contain
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C14: Offers over time in DS-Contain..."

use $mydata, clear
replace allo2b = . if reject1==1
collapse y1 = allo1b y2 = allo2b, by(period treatment contain rival_contain)
twoway (connect y1 period if treatment == "5DS-Contain" & rival_contain==1, m(Dh) lpattern(longdash) lcolor(gs7) mcolor(gs7))	////
		(connect y1 period if treatment == "5DS-Contain" & contain==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	////
		(connect y1 period if treatment == "5DS-Contain" & contain==0 & rival_contain==0, m(Sh) lpattern(solid) lcolor(red) mcolor(red))	////
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)10, labsize(`size') angle(0)) ///
		title("Stage 1") ///
		legend(lab(1 "DS-Contain, contain by DS") lab(2 "DS-Contain, self-contain")  lab(3 "DS-Contain, no action") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/1.gph", replace
twoway (connect y2 period if treatment == "5DS-Contain" & rival_contain==1, m(Dh) lpattern(longdash) lcolor(gs7) mcolor(gs7))	////
		(connect y2 period if treatment == "5DS-Contain" & contain==1, m(Th) lpattern(dash) lcolor(green) mcolor(green))	///
		(connect y2 period if treatment == "5DS-Contain" & contain==0 & rival_contain==0, m(Sh) lpattern(solid) lcolor(red) mcolor(red))	///
		, caption(`caption1', size(`size1') pos(6)) note(`note',ring(0) pos(12) nobox) name(g1,replace)   ///
		xtitle("Round", size(`size')) ytitle("Rising state's offer", size(`size'))  ///
		title("Stage 2") ///
		xlabel(0(2)30, labsize(`size') ) ylabel(0(2)10, labsize(`size') angle(0)) ///
		legend(lab(1 "Contain by DS") lab(2 "Self-contain") lab(3 "No action") size(`size')  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none)))
graph save "$data_folder/temp/2.gph", replace
grc1leg2 "$data_folder/temp/1.gph" "$data_folder/temp/2.gph" , ///
	scheme(s1color) cols(2) lcols(3) pos(6) labsize(small) ytol1title xsize(2) ysize(1) scale(1.3)

graph export "$myfigs/figurec14.png", replace as(png)


*------------------------------------------------------------------------------
// APPENDIX FIGURE C15: Conditional rejection in DS-Contain
*------------------------------------------------------------------------------

display ""
display "Generating FIGURE C15: Conditional rejection in DS-Contain..."

use $mydata, clear
gen count = 1
keep if treatment == "5DS-Contain"
collapse (count) count, by(rival_contain contain allo1b reject1)

bys rival_contain contain allo1b: egen a1 = total(count)
gen rejFreq = count/a1 if reject1 == 1
bys rival_contain contain: egen x = total(count)
keep if reject1 == 1
gen alloFreq = a1/x

gen choice1 = "3No action" if rival_contain==0 & contain==0
replace choice1 = "1Self-contain" if contain == 1
replace choice1 = "2Contain by DS" if rival_contain==1

twoway (bar alloFreq allo1b) ///
	(connect rejFreq allo1b, lpattern(solid) col(black)), ///
	by(choice1, note("") col(3)) ///
	xlabel(0(2)10) ylabel(0(.2)1) ///
	xtitle("Rising state's offer in stage 1") ytitle("Frequency")  ///
	legend(lab(1 "Frequency of stage 1 offer") lab(2 "Frequency of fighting (preventive war)")  ///
		col(1) ring(0) pos(3) region(lcolor(none) fcolor(none))) scheme(s1mono)

graph export "$myfigs/figurec15.png", replace as(png)


//==============================================================================
// END OF ANALYSIS
//==============================================================================

display ""
display "=================== ALL ANALYSIS COMPLETE ==================="
display ""
display "Figures generated in: $myfigs"
display "Tables generated in: $mytables"
display ""
