//==============================================================================
// MASTER ANALYSIS FILE
// "Born to wait? A study on allocation rules in booking systems"
//
// This file processes raw data and generates all figures and tables
// Prerequisite: Run 0PathSetup.do first to set up global paths
//==============================================================================

display ""
display "============================================================================"
display "MASTER ANALYSIS: Processing data and generating all outputs"
display "============================================================================"
display ""

set more off

//==============================================================================
// STEP 1: PROCESS MASTER DATA
//==============================================================================

display ""
display "-----------------------------------------------------------------------------"
display "STEP 1: PROCESSING MASTER DATA"
display "-----------------------------------------------------------------------------"
display ""

display "Loading raw data: $masterdata_raw"
use "$masterdata_raw", clear

//******************************************************************************
// DATA CLEANING - MASTER DATA
//******************************************************************************

// Create group and subject IDs
egen mgroup = group(biggroup sessionid)
gen sid = mgroup * 100 + subject
gen gid = mgroup * 100 + group

// Experiment 1 variables
gen time_val = valuation / (time_cost*60)
gen time_book = time_spent_booking_step1 / 60
replace profit = profit_ct + valuation*slot if _n>2752

gen time_book_per = time_book * 100 /4
gen time_book_t_per = time_book_t * 100 /4

// Demographics
gen female1 = 1 if female=="1"
replace female1 = 0 if female=="0"
bys sessionid subject: egen female2 = max(female1)
gen age1 = real(age)
bys sessionid subject: egen age2 = max(age1)
gen urban1 = real(urban)
bys sessionid subject: egen urban2 = max(urban1)
gen anxiety_general1 = real(anxiety_general)
bys sessionid subject: egen anxiety_general2 = max(anxiety_general1)
gen anxiety_booking1 = real(anxiety_booking)
bys sessionid subject: egen anxiety_booking2 = max(anxiety_booking1)
gen booking_mood1 = real(booking_mood)
bys sessionid subject: egen booking_mood2 = max(booking_mood1)
gen anxiety_step11 = real(anxiety_step1)
bys sessionid subject: egen anxiety_step12 = max(anxiety_step11)
gen risk_taking1 = real(risk_taking)
bys sessionid subject: egen risk_taking2 = max(risk_taking1)
gen competitive1 = real(competitive)
bys sessionid subject: egen competitive2 = max(competitive1)

forvalues i = 1/10 {
  gen bigfive`i'1 = real(bigfive`i')
}
gen extraversion1 = bigfive11 + 8 - bigfive61
gen agreeableness1 = 8 - bigfive21 + bigfive71
gen conscientiousness1 = bigfive31 + 8 - bigfive81
gen stability1 = 8 - bigfive41 + bigfive91
gen openness1 = bigfive51 + 8 - bigfive101
bys sessionid subject: egen extraversion = max(extraversion1)
bys sessionid subject: egen agreeableness = max(agreeableness1)
bys sessionid subject: egen conscientiousness = max(conscientiousness1)
bys sessionid subject: egen stability = max(stability1)
bys sessionid subject: egen openness = max(openness1)

drop female age urban anxiety_general anxiety_booking  ///
booking_mood timequestionnaire1ok anxiety_step1  ///
risk_taking competitive bigfive*  ///
extraversion1 agreeableness1 conscientiousness1 stability1 openness1 ///
female1 age1 urban1 anxiety_general1 anxiety_booking1  ///
booking_mood1 anxiety_step11  ///
risk_taking1 competitive1

rename female2 female
rename age2 age
rename urban2 urban
rename anxiety_general2 anxiety_general
rename anxiety_booking2 anxiety_booking
rename booking_mood2 booking_mood
rename anxiety_step12 anxiety_step1
rename risk_taking2 risk_taking
rename competitive2 competitive

save "$masterdata_clean", replace
display "✓ Saved cleaned master data: $masterdata_clean"
display ""


//==============================================================================
// STEP 2: PROCESS SWITCH DATA (IF AVAILABLE)
//==============================================================================

display ""
display "-----------------------------------------------------------------------------"
display "STEP 2: PROCESSING SWITCH DATA"
display "-----------------------------------------------------------------------------"
display ""

capture confirm file "$switchdata_raw"
if _rc {
    display "⚠ Switch data not found. Skipping switch data processing."
    display "  Some appendix figures/tables may not be generated."
    global has_switch_data 0
}
else {
    display "Loading raw switch data: $switchdata_raw"
    global has_switch_data 1

    use "$switchdata_raw", clear

    // Drop missing booking screens
    drop if booking_screen==-1
    sort treatment sessionid period participant time_login
    bys treatment sessionid period participant: gen number = _n
    bys treatment sessionid period participant: egen switch = max(number)
    replace switch = switch - 1

    // Calculate time spent in booking
    bys sessionid period participant: egen booking = sum(time_switch) if booking_screen==1
    bys sessionid period participant: egen time_booking = max(booking)
    replace time_booking = 0 if time_booking == .

    // Get last booking time
    bys sessionid period participant (time_login): gen n=_n
    bys sessionid period participant: egen nn=max(n)
    keep if n==nn
    gen time_booking_last = time_switch if booking_screen == 1
    replace time_booking_last = 0 if booking_screen == 0

    rename participant subject

    // Merge with master data
    merge 1:1 treatment sessionid period subject using "$masterdata_raw"

    // Create group IDs
    egen mgroup = group(biggroup sessionid)
    gen sid = mgroup * 100 + subject
    gen gid = mgroup * 100 + group

    // Create analysis variables
    gen time_val = valuation / (time_cost*60)
    gen time_book = time_spent_booking_step1 / 60
    gen time_book_last = time_booking_last / 60
    replace profit = profit_ct + valuation*slot if _n>2752

    gen time_book_per = time_book * 100 /4
    gen time_book_t_per = time_book_t * 100 /4
    gen time_book_last_per = time_book_last * 100 /4

    save "$switchdata_clean", replace
    display "✓ Saved cleaned switch data: $switchdata_clean"
    display ""
}

//==============================================================================
// STEP 3: GENERATE MAIN MANUSCRIPT FIGURES AND TABLES
//==============================================================================

display ""
display "-----------------------------------------------------------------------------"
display "STEP 3: GENERATING MAIN MANUSCRIPT OUTPUTS"
display "-----------------------------------------------------------------------------"
display ""

//==============================================================================
// FIGURE 1: Percentage of time spent on the booking system
//==============================================================================

display "Generating FIGURE 1..."
use "$masterdata_clean", clear
collapse y = time_book_per y2 = time_book_t_per, by(treatment mgroup )
ranksum y if treatment=="Queue5" | treatment=="Queue5_rank", by(treatment)
signrank y = y2 if treatment=="Queue5"
signrank y = y2 if treatment=="Queue5_rank"
collapse y = y y2 = y2 (semean) b = y, by(treatment)
gen x = 3 if treatment == "Lottery5"
replace x = 1 if treatment == "Queue5"
replace x = 2 if treatment == "Queue5_rank"
gen y_u = y + b
gen y_l = y - b
gen mean_string = string(y, "%2.1f")

twoway (bar y2 x if treatment == "Queue5", barwidth(.6) bfcolor(gs13)) ///
	(bar y2 x if treatment == "Queue5_rank", barwidth(.6) bfcolor(gs13)) ///
	(bar y x if treatment == "Queue5", barwidth(.6) bfcolor(gs3)) ///
	(bar y x if treatment == "Queue5_rank", barwidth(.6) bfcolor(gs3)) ///
	(bar y x if treatment == "Lottery5", barwidth(.6) bfcolor(gs3)) ///
	(rcap y_u y_l x,lcolor(black)) ///
    (scatter y x, msym(none) mlab(mean_string) mlabsize(medium) mlabpos(1) mlabcolor(black)), ///
	xtitle("") ytitle("Percentage of queuing time", size(small))  ///
	xlabel(1 "Queue5" 2 "Queue5_rank" 3 "Lottery5")  ///
	ylabel(0(20)100) legend(order(1 "Predicted" 3 "Observed") ring(0) pos(11) region(lcolor(none) fcolor(none))) ///
	scheme(s1mono) scale(1.2)

graph export "$figures_folder/figure1.png", replace as(png)
display "✓ Figure 1 saved"


//==============================================================================
// FIGURE 2: Relationship between time valuation and percentage of queuing time
//==============================================================================

display "Generating FIGURE 2..."
use "$masterdata_clean", clear
keep if treatment=="Queue5" | treatment=="Queue5_rank"
lowess time_book_per time_val, by(treatment) scheme(s1mono) xtitle("Time valuation") ytitle("Percentage of queuing time") lineopts(lw(medthick) lp(solod)) xsize(1.7) ysize(1) ///
mcolor(gs12) addplot(line time_book_t_per time_val, sort lp(dash) lw(medthick) lc(black)) legend(order(2 "Observed (lowess smoothing)" 3 "Predicted"))

graph export "$figures_folder/figure2.png", replace as(png)
display "✓ Figure 2 saved"


//==============================================================================
// FIGURE 3: CDF of percentage of total queuing time (two panels)
//==============================================================================

display "Generating FIGURE 3..."

// Left panel: Queue5
use "$masterdata_clean", clear
cumul time_book_per if treatment=="Queue5", gen(cdf1)
cumul time_book_t_per if treatment=="Queue5", gen(cdf2)
stack cdf1 time_book_per cdf2 time_book_t_per, into(c temp) wide clear
line cdf1 cdf2 temp, sort scheme(s1color) xsize(1) ysize(1) xtitle("Percentage of queuing time") ytitle("Cumulative probability") title("Queue5") ///
lc(black black) lp(solid dash) lw(medium medium) legend(lab(1 "Observed") lab(2 "Predicted") ring(0) pos(10) cols(1)) scale(1.2)
graph export "$figures_folder/figure3_left.png", replace as(png)

// Right panel: Queue5_rank
use "$masterdata_clean", clear
cumul time_book_per if treatment=="Queue5_rank", gen(cdf1)
cumul time_book_t_per if treatment=="Queue5_rank", gen(cdf2)
stack cdf1 time_book_per cdf2 time_book_t_per, into(c temp) wide clear
line cdf1 cdf2 temp, sort scheme(s1color) xsize(1) ysize(1) xtitle("Percentage of queuing time") ytitle("Cumulative probability") title("Queue5_rank") ///
lc(black black) lp(solid dash) lw(medium medium) legend(lab(1 "Observed") lab(2 "Predicted") ring(0) pos(10) cols(1)) scale(1.2)
graph export "$figures_folder/figure3_right.png", replace as(png)

display "✓ Figure 3 saved (two panels)"


//==============================================================================
// FIGURE 4: CDF of percentage of effective queuing time (two panels)
//==============================================================================

display "Generating FIGURE 4..."

if $has_switch_data == 1 {
    // Left panel: Queue5
    use "$switchdata_clean", clear
    cumul time_book_last_per if treatment=="Queue5", gen(cdf1)
    cumul time_book_t_per if treatment=="Queue5", gen(cdf2)
    stack cdf1 time_book_last_per cdf2 time_book_t_per, into(c temp) wide clear
    line cdf1 cdf2 temp, sort scheme(s1color) xsize(1) ysize(1) xtitle("Percentage of effective queuing time") ytitle("Cumulative probability") title("Queue5") ///
    lc(black black) lp(solid dash) lw(medium medium) legend(lab(1 "Observed") lab(2 "Predicted") ring(0) pos(10) cols(1)) scale(1.2)
    graph export "$figures_folder/figure4_left.png", replace as(png)

    // Right panel: Queue5_rank
    use "$switchdata_clean", clear
    cumul time_book_last_per if treatment=="Queue5_rank", gen(cdf1)
    cumul time_book_t_per if treatment=="Queue5_rank", gen(cdf2)
    stack cdf1 time_book_last_per cdf2 time_book_t_per, into(c temp) wide clear
    line cdf1 cdf2 temp, sort scheme(s1color) xsize(1) ysize(1) xtitle("Percentage of effective queuing time") ytitle("Cumulative probability") title("Queue5_rank") ///
    lc(black black) lp(solid dash) lw(medium medium) legend(lab(1 "Observed") lab(2 "Predicted") ring(0) pos(10) cols(1)) scale(1.2)
    graph export "$figures_folder/figure4_right.png", replace as(png)

    display "✓ Figure 4 saved (two panels)"
}
else {
    display "⚠ Figure 4 skipped (switch data not available)"
}


//==============================================================================
// TABLE 3: Random effects regressions on time spent in booking
//==============================================================================

display "Generating TABLE 3..."
use "$masterdata_clean", clear
gen time_cost_min = time_cost*60 / 100
gen slot_val = valuation / 100

xtset sid period

eststo: xtreg time_book_per time_val if treatment=="Queue5", cluster(mgroup)
eststo: xtreg time_book_per slot_val time_cost_min if treatment=="Queue5", cluster(mgroup)
eststo: xtreg time_book_per time_val if treatment=="Queue5_rank", cluster(mgroup)
eststo: xtreg time_book_per slot_val time_cost_min if treatment=="Queue5_rank", cluster(mgroup)
eststo: xtreg time_book_per time_val if treatment=="Lottery5", cluster(mgroup)
eststo: xtreg time_book_per slot_val time_cost_min if treatment=="Lottery5", cluster(mgroup)

esttab using "$tables_folder/table3.tex", replace ///
	b(3) se(3) ///
	star(* 0.10 ** 0.05 *** 0.01) ///
	nomtitles nonumbers compress ///
	label width(0.6\hsize)

eststo clear
display "✓ Table 3 saved"


//==============================================================================
// TABLE 4: Random effects probit regressions on likelihood of obtaining a slot
// ONLY REPORTING MARGINAL EFFECTS
//==============================================================================

display "Generating TABLE 4..."
use "$masterdata_clean", clear
gen time_cost_min = time_cost*60 / 100
gen slot_val = valuation / 100

xtset sid period

// Queue5 - time_val
quietly xtprobit slot time_val if treatment=="Queue5", vce(cluster mgroup)
est store t4m1
quietly margins, predict(pu0) dydx(*) post
est store t4m1_margins

// Queue5 - slot_val and time_cost_min
quietly xtprobit slot slot_val time_cost_min if treatment=="Queue5", vce(cluster mgroup)
est store t4m2
quietly margins, predict(pu0) dydx(*) post
est store t4m2_margins

// Queue5_rank - time_val
quietly xtprobit slot time_val if treatment=="Queue5_rank", vce(cluster mgroup)
est store t4m3
quietly margins, predict(pu0) dydx(*) post
est store t4m3_margins

// Queue5_rank - slot_val and time_cost_min
quietly xtprobit slot slot_val time_cost_min if treatment=="Queue5_rank", vce(cluster mgroup)
est store t4m4
quietly margins, predict(pu0) dydx(*) post
est store t4m4_margins

// Lottery5 - time_val
quietly xtprobit slot time_val if treatment=="Lottery5", vce(cluster mgroup)
est store t4m5
quietly margins, predict(pu0) dydx(*) post
est store t4m5_margins

// Lottery5 - slot_val and time_cost_min
quietly xtprobit slot slot_val time_cost_min if treatment=="Lottery5", vce(cluster mgroup)
est store t4m6
quietly margins, predict(pu0) dydx(*) post
est store t4m6_margins

esttab t4m1_margins t4m2_margins t4m3_margins t4m4_margins t4m5_margins t4m6_margins using "$tables_folder/table4.tex", replace ///
	b(3) se(3) ///
	star(* 0.10 ** 0.05 *** 0.01) ///
	nomtitles nonumbers compress ///
	label width(0.6\hsize)

eststo clear
display "✓ Table 4 saved (marginal effects only)"


//==============================================================================
// TABLE 5: Random effects probit regressions on bimodal behavior
// ONLY REPORTING MARGINAL EFFECTS
//==============================================================================

display "Generating TABLE 5..."
use "$masterdata_clean", clear
xtset sid period

gen high_time = (time_spent_booking_step1 >= 235)
gen low_time = (time_spent_booking_step1 <=5)

// Queue5 - high_time
quietly xtprobit high_time time_val if treatment=="Queue5", vce(cluster mgroup)
est store t5m1
quietly margins, predict(pu0) dydx(*) post
est store t5m1_margins

// Queue5 - low_time
quietly xtprobit low_time time_val if treatment=="Queue5", vce(cluster mgroup)
est store t5m2
quietly margins, predict(pu0) dydx(*) post
est store t5m2_margins

// Queue5_rank - high_time
quietly xtprobit high_time time_val if treatment=="Queue5_rank", vce(cluster mgroup)
est store t5m3
quietly margins, predict(pu0) dydx(*) post
est store t5m3_margins

// Queue5_rank - low_time
quietly xtprobit low_time time_val if treatment=="Queue5_rank", vce(cluster mgroup)
est store t5m4
quietly margins, predict(pu0) dydx(*) post
est store t5m4_margins

esttab t5m1_margins t5m2_margins t5m3_margins t5m4_margins using "$tables_folder/table5.tex", replace ///
	b(3) se(3) ///
	star(* 0.10 ** 0.05 *** 0.01) ///
	nomtitles nonumbers compress ///
	label width(0.6\hsize)

eststo clear
display "✓ Table 5 saved (marginal effects only)"


//==============================================================================
// TABLE 6: The Breakdown of Efficiency Loss
//==============================================================================

display "Generating TABLE 6..."
use "$masterdata_clean", clear
bys gid period: egen actual_match = sum(valuation) if slot==1
bys gid period: egen actual_match_sum = max(actual_match)
bys gid period (valuation): gen order = _n
bys gid period: egen optimal_match = sum(valuation) if order > 2
bys gid period: egen optimal_match_sum = max(optimal_match)
gen mismatch_sum =  optimal_match_sum - actual_match_sum

bys gid period: egen productive = sum(time_cost * time_spent_booking_step1)
bys gid period: egen productive_sum = max(productive)
gen total = mismatch_sum + productive_sum

display ""
display "TABLE 6: BREAKDOWN OF EFFICIENCY LOSS"
display "--------------------------------------"
bys treatment: ci means mismatch_sum productive_sum total if order==1
display "--------------------------------------"
display ""

// Save Table 6 as LaTeX table
preserve
keep if order==1

// Calculate means and standard errors for each treatment
collapse (mean) mean_mismatch=mismatch_sum mean_productive=productive_sum mean_total=total (sd) sd_mismatch=mismatch_sum sd_productive=productive_sum sd_total=total (count) n=mismatch_sum, by(treatment)

// Calculate standard errors
gen se_mismatch = sd_mismatch / sqrt(n)
gen se_productive = sd_productive / sqrt(n)
gen se_total = sd_total / sqrt(n)

// Sort treatments
gen sort_order = 1 if treatment=="Queue5"
replace sort_order = 2 if treatment=="Queue5_rank"
replace sort_order = 3 if treatment=="Lottery5"
sort sort_order

// Create LaTeX table
file open table6 using "$tables_folder/table6.tex", write replace
file write table6 "\begin{tabular}{lccc}" _n
file write table6 "\toprule" _n
file write table6 "Treatment & Allocative Efficiency Loss & Productive Efficiency Loss & Total Efficiency Loss \\" _n
file write table6 "\midrule" _n

forvalues i = 1/`=_N' {
    local trt = treatment[`i']
    local alloc = string(mean_mismatch[`i'], "%6.2f")
    local alloc_se = string(se_mismatch[`i'], "%6.2f")
    local prod = string(mean_productive[`i'], "%6.2f")
    local prod_se = string(se_productive[`i'], "%6.2f")
    local tot = string(mean_total[`i'], "%6.2f")
    local tot_se = string(se_total[`i'], "%6.2f")

    file write table6 "`trt' & `alloc' (`alloc_se') & `prod' (`prod_se') & `tot' (`tot_se') \\" _n
}

file write table6 "\bottomrule" _n
file write table6 "\end{tabular}" _n
file close table6

restore

display "✓ Table 6 saved as LaTeX table"


//==============================================================================
// STEP 4: GENERATE APPENDIX B FIGURES AND TABLES
//==============================================================================

display ""
display "-----------------------------------------------------------------------------"
display "STEP 4: GENERATING APPENDIX B OUTPUTS"
display "-----------------------------------------------------------------------------"
display ""

//==============================================================================
// FIGURE B1: The evolution of percentage of time spent on booking
//==============================================================================

display "Generating FIGURE B1..."
use "$masterdata_clean", clear
bys treatment period: egen av_book = mean(time_book *100 /4)
twoway (connected av_book period if treatment=="Queue5", sort m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick)) ///
(connected av_book period if treatment=="Queue5_rank", sort m(S) lpattern(dash) lcolor(black) mcolor(black)) ///
(connected av_book period if treatment=="Lottery5", sort m(S) lpattern(solid) lcolor(gs13) mcolor(gs13) lwidth(thick)) ///
, legend(lab(1 "Queue5") lab(2 "Queue5_rank") lab(3 "Lottery5") ///
col(3) ring(0) pos(11) region(lcolor(none) fcolor(none))) scheme(s1mono) ///
xlabel(1(1)8, labsize(small) ) ylabel(0(20)100, labsize(small) angle(0)) ///
xtitle("Round", size(small)) ytitle("Percentage")

graph export "$figures_folder/figureB1.png", replace as(png)
display "✓ Figure B1 saved"


//==============================================================================
// FIGURE B2: The evolution of effective and ineffective queuing time
//==============================================================================

display "Generating FIGURE B2..."

if $has_switch_data == 1 {
    use "$switchdata_clean", clear
    bys treatment period: egen av_book = mean(time_book *100 /4)
    bys treatment period: egen av_book_last = mean(time_book_last *100 /4)
    gen av_book_waste = av_book - av_book_last

    twoway (connected av_book_last period if treatment=="Queue5_rank", sort m(S) lpattern(solid) lcolor(black) mcolor(black) lwidth(thick)) ///
    (connected av_book_waste period if treatment=="Queue5_rank", sort m(S) lpattern(dash) lcolor(black) mcolor(black)) ///
    , legend(lab(1 "Effective queuing") lab(2 "Ineffective queuing") ///
    col(2) ring(0) pos(11) region(lcolor(none) fcolor(none))) scheme(s1mono) ///
    xlabel(1(1)8, labsize(small) ) ylabel(0(20)100, labsize(small) angle(0)) ///
    xtitle("Round", size(small)) ytitle("Percentage")

    graph export "$figures_folder/figureB2.png", replace as(png)
    display "✓ Figure B2 saved"
}
else {
    display "⚠ Figure B2 skipped (switch data not available)"
}


//==============================================================================
// TABLE B1: Switching frequency regressions
//==============================================================================

display "Generating TABLE B1..."

if $has_switch_data == 1 {
    use "$switchdata_clean", clear
    gen high_switch = (switch>12)
    xtset sid period

    eststo: xtreg switch time_val if treatment=="Queue5_rank", cluster(mgroup)

    esttab using "$tables_folder/tableB1.tex", replace ///
        b(3) se(3) ///
        star(* 0.10 ** 0.05 *** 0.01) ///
        nomtitles nonumbers compress ///
        label width(0.6\hsize)

    eststo clear
    display "✓ Table B1 saved"
}
else {
    display "⚠ Table B1 skipped (switch data not available)"
}


//==============================================================================
// TABLE B2: Effective queuing time regressions
//==============================================================================

display "Generating TABLE B2..."

if $has_switch_data == 1 {
    use "$switchdata_clean", clear
    gen time_cost_min = time_cost*60 / 100
    gen slot_val = valuation / 100

    xtset sid period

    eststo: xtreg time_book_last_per time_val if treatment=="Queue5", cluster(mgroup)
    eststo: xtreg time_book_last_per slot_val time_cost_min if treatment=="Queue5", cluster(mgroup)
    eststo: xtreg time_book_last_per time_val if treatment=="Queue5_rank", cluster(mgroup)
    eststo: xtreg time_book_last_per slot_val time_cost_min if treatment=="Queue5_rank", cluster(mgroup)

    esttab using "$tables_folder/tableB2.tex", replace ///
        b(3) se(3) ///
        star(* 0.10 ** 0.05 *** 0.01) ///
        nomtitles nonumbers compress ///
        label width(0.6\hsize)

    eststo clear
    display "✓ Table B2 saved"
}
else {
    display "⚠ Table B2 skipped (switch data not available)"
}


//==============================================================================
// COMPLETION MESSAGE
//==============================================================================

display ""
display "============================================================================"
display "ANALYSIS COMPLETE!"
display "============================================================================"
display ""
display "All outputs have been generated in:"
display "  Figures: $figures_folder"
display "  Tables:  $tables_folder"
display ""
display "Summary of generated files:"
display ""
display "MAIN MANUSCRIPT:"
display "  Figure 1: figure1.png"
display "  Figure 2: figure2.png"
display "  Figure 3: figure3_left.png, figure3_right.png"
display "  Figure 4: figure4_left.png, figure4_right.png (if switch data available)"
display "  Table 3:  table3.tex"
display "  Table 4:  table4.tex (marginal effects only)"
display "  Table 5:  table5.tex (marginal effects only)"
display "  Table 6:  table6_summary.txt"
display ""
display "APPENDIX B:"
display "  Figure B1: figureB1.png"
display "  Figure B2: figureB2.png (if switch data available)"
display "  Table B1:  tableB1.tex (if switch data available)"
display "  Table B2:  tableB2.tex (if switch data available)"
display ""
display "Cleaned data files:"
display "  $masterdata_clean"
if $has_switch_data == 1 {
    display "  $switchdata_clean"
}
display ""
display "============================================================================"
display ""
